<?php

defined('ABSPATH') or die('No script kiddies please!!');
/**
 * Plugin Name:       Article Read Time
 * Plugin URI:        https://wpshuffle.com/article-read-time
 * Description:       This plugin will calculate the time to read the article or news.
 * Version:           1.0.3
 * Requires at least: 5.5
 * Requires PHP:      5.0
 * Author:            WP Shuffle
 * Author URI:        https://wpshuffle.com/       
 * Text Domain:       article-read-time
 * Domain Path:       /languages
 */

if (!class_exists('Article_Read_Time')) {

    class Article_Read_Time {

        function __construct() {
            $this->define_constants();
            add_action('plugins_loaded', array($this, 'load_art_textdomain'));
            add_action('admin_menu', array($this, 'art_admin_menu'));
            add_action('admin_enqueue_scripts', array($this, 'art_admin_assets'));
            add_action('wp_enqueue_scripts', array($this, 'art_frontend_assets'));
            add_action('admin_post_art_settings_save_action', array($this, 'save_settings_action'));
            add_action('admin_post_art_progress_bar_settings_save_action', array($this, 'save_progress_bar_settings_action'));
            add_filter('the_content', array($this, 'if_art_post'));
            add_action('wp_footer', array($this, 'if_art_progress_bar'));
            add_action('add_meta_boxes', array($this, 'art_meta_boxes'));
            add_action('save_post', array($this, 'art_save_meta_box'), 10, 2);
        }

        function load_art_textdomain() {
            load_plugin_textdomain('article-read-time', false, dirname(plugin_basename(__FILE__)) . '/languages');
        }
        function define_constants() {
            defined('ART_PATH') or define('ART_PATH', plugin_dir_path(__FILE__));
            defined('ART_CSS_DIR') or define('ART_CSS_DIR', untrailingslashit(plugin_dir_url(__FILE__)) . '/assets/css'); // plugin's CSS directory URL
            defined('ART_IMG_DIR') or define('ART_IMG_DIR', plugin_dir_url(__FILE__) . 'images');
            defined('ART_URL') or define('ART_URL', plugin_dir_url(__FILE__));
            defined('ART_VERSION') or define('ART_VERSION', '1.0.3');
        }

        function art_meta_boxes() {
            add_meta_box('art-meta-box', esc_html__('Article Read Time', 'article-read-time'), array($this, 'art_metabox'), array('post', 'page'), 'side');
        }

        function art_metabox($post) {
            $art = get_post_meta($post->ID, 'art_post', true);

            wp_nonce_field('art_settings_post_nonce', 'art_settings_post_nonce_field');
            $art_post_status = (!empty($art['status'])) ? $art['status'] : '';

            $art_post_time = (!empty($art['custom_time'])) ? $art['custom_time'] : '';
?>

            <div class="art-field-wrap">
                <label><?php esc_html_e('Disable Status', 'article-read-time'); ?></label>
                <div class="art-field art-checkbox-toggle">
                    <input type="checkbox" id="art-basic-status" name="art_post[status]" value="1" <?php checked($art_post_status, '1'); ?> />
                    <label></label>
                </div>
            </div>
            <div class="art-field-wrap">
                <label><?php esc_html_e('Time to Read Article', 'article-read-time'); ?></label>
                <div class="art-field">

                    <input type="number" name="art_post[custom_time]" value="<?php echo esc_attr($art_post_time); ?>" />
                    <p class="description"><?php echo esc_html__('Please enter the custom time in minutes', 'article-read-time'); ?></p>

                </div>
            </div>
            <style>
                .art-field-wrap label {
                    font-weight: 500;
                    margin-bottom: 5px;
                    display: block;
                }

                .art-field-wrap {
                    margin-bottom: 10px;
                }
            </style>
<?php
        }
        function art_save_meta_box($post_id, $post) {

            if (
                !empty($_POST['art_settings_post_nonce_field']) &&
                wp_verify_nonce($_POST['art_settings_post_nonce_field'], 'art_settings_post_nonce')
            ) {

                $post_status = (isset($_POST['art_post']['status'])) ? sanitize_text_field($_POST['art_post']['status']) : '';
                $custom_time = (isset($_POST['art_post']['custom_time'])) ? sanitize_text_field($_POST['art_post']['custom_time']) : '';
                $art_meta_values = array(

                    'status' => $post_status,
                    'custom_time' => $custom_time

                );
                update_post_meta($post_id, 'art_post', $art_meta_values);
            }
        }

        function if_art_progress_bar() {
            wp_enqueue_style('art-frontend-progressbar', ART_CSS_DIR . '/art-progressbar.css', array(), ART_VERSION);
            if (is_single() or is_page()) {
                global $post;

                $progress_bar_check_post_type = $post->post_type;

                $art_progress_bar_settings = get_option('art_progress_bar_settings');
                /**
                 * Filters settings before rendering Article Read Time Block
                 * 
                 * @param array $art_progress_bar_settings
                 * @since 1.0.0
                 */
                $art_progress_bar_settings = apply_filters('art_frontend_settings', $art_progress_bar_settings);

                $progress_bar_post_type = (!empty($art_progress_bar_settings['progress-bar']['display_post_types_progress_bar'])) ? $art_progress_bar_settings['progress-bar']['display_post_types_progress_bar'] : [];

                if ((!empty($art_progress_bar_settings['progress-bar']['status'])) && in_array($progress_bar_check_post_type, $progress_bar_post_type)) {

                    global $wp_query;
                    $post_slug = (!empty($wp_query->queried_object->post_name)) ? $wp_query->queried_object->post_name : '';

                    if (!empty($art_progress_bar_settings['progress-bar']['display_specific'][$progress_bar_check_post_type])) {
                        $allowed_slugs = explode(',', $art_progress_bar_settings['progress-bar']['display_specific'][$progress_bar_check_post_type]);
                        if (!in_array($post_slug, $allowed_slugs)) {
                            return;
                        }
                    }

                    $position = esc_attr($art_progress_bar_settings['progress-bar']['bar_display_position']);
                    $d_style = esc_attr($art_progress_bar_settings['progress-bar']['display_style']);
                    $background = esc_attr($art_progress_bar_settings['progress-bar']['background_color']);
                    $primary_color = esc_attr($art_progress_bar_settings['progress-bar']['primary_color']);
                    $secondary_color = esc_attr($art_progress_bar_settings['progress-bar']['secondary_color']);
                    $bar_thickness = esc_attr($art_progress_bar_settings['progress-bar']['bar_thickness']);
                    $gr_style = esc_attr($art_progress_bar_settings['progress-bar']['display_gradient_style']);
                    echo '<div class="art-progressbar-header"><div class="art-progress-container">
                    <div class="art-progress-bar" id="artProgressBar">
                    </div>
                    </div></div>';
                    $alias_class_1 = 'art-progressbar-header';
                    $alias_class_2 = 'art-progress-container';
                    $alias_class_3 = 'art-progress-bar';

                    include(ART_PATH . '/includes/cores/progress-bar-customize.php');
                }
            }
        }
        function if_art_post($content) {
            wp_enqueue_style('art-frontend-custom', ART_CSS_DIR . '/art-custom.css', array(), ART_VERSION);
            global $post;

            $art_post_settings = get_post_meta($post->ID, 'art_post', true);

            /**
             * If status is disabled from the post edit screen
             */
            if (!empty($art_post_settings['status'])) {
                return $this->createPlainHtml($content);
            }
            $check_post_type = $post->post_type;
            $art_settings = get_option('art_settings');

            if (!empty($art_settings['basic']['status']) && !empty($art_settings['basic']['display_post_types'])) {
                $art_post_type = $art_settings['basic']['display_post_types'];
                if (in_array($check_post_type, $art_post_type)  && $art_settings['basic']['status'] == 1) {
                    if (
                        is_single() or is_page()
                    ) {
                        global $wp_query;
                        $post_slug = (!empty($wp_query->queried_object->post_name)) ? $wp_query->queried_object->post_name : '';

                        if (!empty($art_settings['basic']['display_specific'][$check_post_type])) {
                            $allowed_slugs = explode(',', $art_settings['basic']['display_specific'][$check_post_type]);
                            if (!in_array($post_slug, $allowed_slugs)) {
                                return $content;
                            }
                        }
                        return $this->createHTMLPost($content);
                    }
                }
            }
            return $content;
        }
        function createHTMLPost($content) {
            ob_start();
            include(ART_PATH . 'includes/views/frontend/posthtml.php');
            $html = ob_get_contents();
            ob_end_clean();
            if ($art_settings['layout']['display_section'] == 'beginning') {
                return $html . $content;
            } elseif ($art_settings['layout']['display_section'] == 'both') {
                return  $html . $content . $html;
            }
            return $content . $html;
        }

        function createPlainHtml($content) {

            return $content;
        }


        function art_admin_assets() {
            if (
                isset($_GET['page']) && isset($_GET['page']) == 'article-read-time' ||
                isset($_GET['page']) == 'art-progress-bar' || isset($_GET['page']) == 'art-help' || isset($_GET['page']) == 'art-about'
            ) {
                wp_enqueue_style('art-backend-style', ART_URL . 'assets/css/art-backend.css', array(), ART_VERSION);


                wp_enqueue_script('art-backend-script', ART_URL . 'assets/js/art-backend.js', array('jquery', 'wp-color-picker'), ART_VERSION);
            }
        }

        function art_frontend_assets() {

            wp_enqueue_style('art-frontend-style', ART_URL . 'assets/css/art-frontend.css', array(), ART_VERSION);
            wp_enqueue_script('art-frontend-script', ART_URL . 'assets/js/art-frontend.js', array('jquery'), ART_VERSION);
        }


        function art_admin_menu() {
            add_menu_page(esc_html__('Article Read Time', 'article-read-time'), esc_html__('Article Read Time', 'article-read-time'), 'manage_options', 'article-read-time', array($this, 'art_settings_page'), 'dashicons-clock');

            add_submenu_page('article-read-time', esc_html__('Article Read Time Settings', 'article-read-time'), esc_html__('Article Read Time Settings', 'article-read-time'), 'manage_options', 'article-read-time', array($this, 'art_settings_page'));
            add_submenu_page('article-read-time', esc_html__('Progress Bar', 'article-read-time'), esc_html__('Progress Bar', 'article-read-time'), 'manage_options', 'art-progress-bar', array($this, 'art_progress_bar_page'));
            add_submenu_page('article-read-time', esc_html__('About', 'article-read-time'), esc_html__('About', 'article-read-time'), 'manage_options', 'art-about', array($this, 'art_about'));
            add_submenu_page('article-read-time', esc_html__('Help', 'article-read-time'), esc_html__('Help', 'article-read-time'), 'manage_options', 'art-help', array($this, 'art_help'));
        }

        function art_progress_bar_page() {
            include(ART_PATH . 'includes/views/backend/progress-bar-settings.php');
        }

        function art_settings_page() {
            include(ART_PATH . 'includes/views/backend/settings.php');
        }

        function art_help() {
            include(ART_PATH . 'includes/views/backend/help.php');
        }

        function art_about() {
            include(ART_PATH . 'includes/views/backend/about.php');
        }



        function save_settings_action() {

            if (!empty($_POST['art_settings_nonce_field']) && wp_verify_nonce($_POST['art_settings_nonce_field'], 'art_settings_nonce')) {

                $art_settings = wp_unslash($_POST['art_settings']);
                $art_settings = $this->sanitize_array($art_settings);
                update_option('art_settings', $art_settings);
                wp_redirect(admin_url('admin.php?page=article-read-time&message=1'));
                exit;
            }
        }
        function save_progress_bar_settings_action() {
            if (!empty($_POST['art_progress_bar_settings_nonce_field']) && wp_verify_nonce($_POST['art_progress_bar_settings_nonce_field'], 'art_progress_bar_settings_nonce')) {
                $art_progress_bar_settings = $this->sanitize_array(wp_unslash($_POST['art_progress_bar_settings']));
                update_option('art_progress_bar_settings', $art_progress_bar_settings);
                wp_redirect(admin_url('admin.php?page=art-progress-bar&message=1'));
                exit;
            }
        }

        /**
         * Sanitizes Multi Dimensional Array
         * @param array $array
         * @param array $sanitize_rule
         * @return array
         *
         * @since 1.0.0
         */
        function sanitize_array($array = array(), $sanitize_rule = array()) {
            if (!is_array($array) || count($array) == 0) {
                return array();
            }

            foreach ($array as $k => $v) {
                if (!is_array($v)) {

                    $default_sanitize_rule = (is_numeric($k)) ? 'html' : 'text';
                    $sanitize_type = isset($sanitize_rule[$k]) ? $sanitize_rule[$k] : $default_sanitize_rule;
                    $array[$k] = $this->sanitize_value($v, $sanitize_type);
                }
                if (is_array($v)) {
                    $array[$k] = $this->sanitize_array($v, $sanitize_rule);
                }
            }

            return $array;
        }

        /**
         * Sanitizes Value
         *
         * @param type $value
         * @param type $sanitize_type
         * @return string
         *
         * @since 1.0.0
         */
        function sanitize_value($value = '', $sanitize_type = 'text') {
            switch ($sanitize_type) {
                case 'html':
                    return wp_kses_post($value);
                    break;
                case 'to_br':
                    return $this->sanitize_escaping_linebreaks($value);
                    break;
                default:
                    return sanitize_text_field($value);
                    break;
            }
        }


        /**
         * Sanitizes field by converting line breaks to <br /> tags
         *
         * @since 1.0.0
         *
         * @return string $text
         */
        function sanitize_escaping_linebreaks($text) {
            $text = implode("<br \>", array_map('sanitize_text_field', explode("\n", $text)));
            return $text;
        }
    }

    new Article_Read_Time();
}
